import javax.servlet.*;
import javax.servlet.http.*;

import java.io.*;
import java.util.*;

import com.wrox.context.chat.*;
import com.wrox.util.StringUtils;


public class ChatAdminServlet extends HttpServlet
{

   public void init() throws ServletException
   {
      String propsFile = getInitParameter("chatprops");
      if (propsFile == null || propsFile.length() == 0)
      {
         throw new UnavailableException(this, "chatprops not set in servlet init parameters");
      }

      Properties props = new Properties();

      try
      {
         InputStream is = new FileInputStream(propsFile);
         props.load(is);
         is.close();
      }
      catch (Exception e)
      {
         throw new UnavailableException(this, "Can't read the chatprops file " +
               propsFile);
      }

      RoomList roomList = createRooms(props);
      getServletContext().setAttribute("roomList", roomList);
   }

   public void doGet(HttpServletRequest req, HttpServletResponse res) 
                  throws IOException, ServletException
   {
      res.setContentType("text/html");
      PrintWriter out = res.getWriter();
      writePage(out);
      out.close();
   }

   public void doPost(HttpServletRequest req, HttpServletResponse res) 
            throws IOException, ServletException
   {
        
      boolean isListModified = false;
      RoomList roomList = 
         (RoomList) getServletContext().getAttribute("roomList");

      // Update the room list
      String[] removeList = req.getParameterValues("remove");
      if (removeList != null)
      {
         roomList.removeRooms(removeList);
         isListModified = true;
      }
      
      String roomName = req.getParameter("roomname");
      String roomDescr = req.getParameter("roomdescr");
      if (roomName != null && roomName.length() > 0)
      {
         roomList.addRoom(new ChatRoom(roomName, roomDescr));
         isListModified = true;
      }

      if (isListModified)
      {
         saveList(roomList);
      }

      doGet(req, res);
   }

   private void saveList(RoomList roomList) throws UnavailableException
   {
      String propsFile = getServletConfig().getInitParameter("chatprops");

      if (propsFile == null || propsFile.length() == 0)
      {
         throw new UnavailableException(this, "chatprops not set");
      }
        
      Properties props = new Properties();
      Enumeration rooms = roomList.getRooms();
      while (rooms.hasMoreElements())
      {
         ChatRoom room = (ChatRoom) rooms.nextElement();
         String roomName = StringUtils.replaceInString(room.getName(), " ", "+");
         props.put(roomName + ".descr", room.getDescription());
      }
      try
      {
         OutputStream os = new FileOutputStream(propsFile);
         props.store(os, "Generated by ChatAdminServlet");
         os.close();
      }
      catch (Exception e)
      {
         throw new UnavailableException(this, "Can't write the chatprops file " +
            propsFile);
      }
   }

   private RoomList createRooms(Properties props)
   {
      RoomList roomList = new RoomList();
      Enumeration propKeys = props.keys();
      while (propKeys.hasMoreElements())
      {
         String key = (String) propKeys.nextElement();
         if (key.endsWith(".description"))
         {
            String roomDescription = props.getProperty(key);
            String roomName = key.substring(0, key.lastIndexOf("."));
            roomName = StringUtils.replaceInString(roomName, "+", " ");
            roomList.addRoom(new ChatRoom(roomName, roomDescription));
         }
      }
      return roomList;
   }

   private void writePage(PrintWriter out)
   {
      out.println("<HTML>");
      out.println("<HEAD><TITLE>Chat room administration</TITLE></HEAD>");
      out.println("<BODY>");
      out.println("<H1>Chat room administration</H1>");
      out.println("<FORM METHOD=POST ACTION=chatAdmin>");
        
      // Add check boxes for removing rooms
      out.println("Check off the rooms you would like to remove:<P>");
      RoomList roomList = (RoomList) getServletContext().getAttribute("roomList");
      Enumeration rooms = roomList.getRooms();
      while (rooms.hasMoreElements())
      {
         ChatRoom room = (ChatRoom) rooms.nextElement();
         out.println("<INPUT TYPE=CHECKBOX NAME=remove VALUE='" +
                  room.getName() + "'>" + room.getName() + "<BR>");
      }
      // Add fields for adding a room
      out.println("<P>Describe the room you would like to add:<P>");
      out.println("<TABLE>");
      out.println("<TR><TD>Name:</TD><TD><INPUT NAME=roomname SIZE=50></TD></TR>");
      out.println("<TR><TD>Description:</TD>");
      out.println("<TD><TEXTAREA NAME=roomdescr COLS=50 ROWS=15>");
      out.println("</TEXTAREA></TD></TR>");
      out.println("</TABLE>");
        
      // Add submit button
      out.println("<P><INPUT TYPE=SUBMIT VALUE='Update List'>");

      out.println("</FORM>");
      out.println("</BODY></HTML>");
   }
}