//Source file: c:/work/wrox/case_study/code/BugTrackerApp/businesslogic/DefectPolicy.java

package businesslogic;
import businessmodel.*;
import dataabstraction.*;

import java.util.Date;
import java.sql.SQLException;
import java.io.*;
import java.net.*;

/**
Singleton class that enforces the policies of the business objects
of the BugTracker application. This involves primarily the creation and
modification of Defects and Users. Implements the _DefectPolicyIntfOperations 
interface used by the tie class generated by javatoidl
*/
public class DefectPolicy implements _DefectPolicyIntfOperations , apputil.Testable
{
	
	int m_nextDefectID;
	org.omg.CORBA.ORB m_Orb;
	
	
	public DefectPolicy() 
	{
		// make a defect to get the max id
		m_Orb = null;

		PersistentDefect theDefect = new PersistentDefect();
		try
		{
			m_nextDefectID = theDefect.getMaxDefectID() + 1;
		}
		catch (SQLException sqle)
		{
			System.out.println("WARNING: Error determining defect id sequence. Using 0 base");
			m_nextDefectID = 1;
		}
	}
	
	/**
	   creates a new defect object. This method is called by the DefectFactory,
	   thus allowing the DefectPolicy object to enforce creation policies
	   where they may exist.
	   @param uid users identifier
	   @return returns the new DefectIntf - implementing objecf
	  */
	
	public businessmodel.DefectIntf createNewDefect(String uid)
	{
		businessmodel._DefectIntfTie base;
		
		// need to get assign a new uid to the defect
		PersistentDefect theDefect = 	new PersistentDefect();
		theDefect.setDefectID(getNextDefectID());
		theDefect.setPolicy(this);
		
		Date rightNow = new Date();
		theDefect.setDateReported(rightNow.getTime());
		
		base = new businessmodel._DefectIntfTie(theDefect);
		implConnect(base);

		return base;
		
   }
  
  /**
  Provides a factory method for creating new User objects. Calling this
  method requires that the requesting user have administrator privileges. if
  sufficient privileges are not held, an AppSecurityException will be thrown.
  
  Once this method has been successfully called, the client application can 
  populate the User object via the setXXXX methods, and write it to the 
  database with the commit() method.
  @param requestor id of the user making the request to create a new user
  @param userid identification for the new user. This must be unique; otherwise this method will throw an AlreadyExistsException
  @exception AlreadyExistsException thrown if the userid already exists
  @exception AppSecurityException thrown if the requesting user does not have sufficient privileges
  */
   
  public businessmodel.UserIntf createNewUser(String requestor, String userid) throws AlreadyExistsException, AppSecurityException
	{
		
		if (!canAdminister(requestor))
			throw new AppSecurityException();
		
		businessmodel._UserIntfTie base;
		PersistentUser pUser = new PersistentUser();
		pUser.setUserID(userid);
		if (pUser.rowExists())
			throw new AlreadyExistsException();
		
		base = new businessmodel._UserIntfTie(pUser);
		implConnect(base);
		
		return base;
		
		
	}
 
	
	/**
	   when the defect has changed, the policy object will notify the list of addressees (in the changedMailList member of the defect)  that the defect has changed, and include the updated defect.
	   
	   If the defect has been set to FIXED or FIX_VERIFIED, the policy object will notify the list of addressees indicated in the defect's resolvedMailList member
	   
	 */
	public void statusChanged( _DefectIntfOperations theDefect) throws SecurityException 
	{
		try
		{
			// minimally, send notification to the changed list
			MailMessage msg = new MailMessage();
			msg.setRecipient(theDefect.getChangedMailList());
			msg.setMessage("Subject: Defect  " + theDefect.getDefectID() + 
				" updated\n" + theDefect.toString());
				
			msg.send();	
				
			// was the defect set to FIXED or FIX_VERIFIED?
			int state = theDefect.getState();
			if ((state == DefectState.FIXED) || (state == DefectState.FIX_VERIFIED))
			{
				// send to resolved list
				msg.setRecipient(theDefect.getResolvedMailList());
				msg.send();
			}
		}
		catch (UnknownHostException uhe)
		{
			System.err.println(uhe.toString());
		}
		catch (IOException ioe)
		{
			System.err.println(ioe.toString());
		}

	}
		
		
	/**
	 Determines if the specified user holds privileges for assigning defects
	 @param userid id of the user
	 @return true if user holds sufficient privileges, false otherwise
	 */
	public boolean canAssign(String userid) 
	{
		boolean retval = false;
		
		try
		{
				PersistentUser user = new PersistentUser(userid);
				user.fetch();
				int userpriv = user.getPriv();
				if (0 != (userpriv & DefectPrivileges.CANCHANGEASSIGN))
					retval = true;
		}
		catch (SQLException sqle)
		{
			;// we'll just return false
		}
		
		return retval;

	}
	
	/**
	 Determines if the specified user holds privileges for prioritizing defects
	 @param userid id of the user
	 @return true if user holds sufficient privileges, false otherwise
	 */
	public boolean canPrioritize(String userid) 
	{
		boolean retval = false;
		
		try
		{
				PersistentUser user = new PersistentUser(userid);
				user.fetch();
				int userpriv = user.getPriv();
				if (0 != (userpriv & DefectPrivileges.CANCHANGEPRIORITY))
					retval = true;
		}
		catch (SQLException sqle)
		{
			;// we'll just return false
		}
		
		return retval;
		
	}
	
 /**
	 Determines if the specified user holds privileges for marking fixed defects as "verified"
	 @param userid id of the user
	 @return true if user holds sufficient privileges, false otherwise
	 */
	public boolean canVerify(String userid) 
	{
		boolean retval = false;
		
		try
		{
				PersistentUser user = new PersistentUser(userid);
				user.fetch();
				int userpriv = user.getPriv();
				if (0 != (userpriv & DefectPrivileges.CANVERIFY))
					retval = true;
		}
		
		catch (SQLException sqle)
		{
			;// we'll just return false
		}
		
		return retval;
		
	}
	
	/**
	 Determines if the specified user holds privileges for marking defects as fixed
	 @param userid id of the user
	 @return true if user holds sufficient privileges, false otherwise
	 */
	public boolean canResolve(String userid) 
	{
		boolean retval = false;
		
		try
		{
				PersistentUser user = new PersistentUser(userid);
				user.fetch();
				int userpriv = user.getPriv();
				if (0 !=(userpriv & DefectPrivileges.CANRESOLVE))
					retval = true;
		}
		
		catch (SQLException sqle)
		{
			;// we'll just return false
		}
		
		return retval;
		
	}
	
	/**
	 Determines if the specified user holds administrative privileges for the BugTracker application
	 @param userid id of the user
	 @return true if user holds sufficient privileges, false otherwise
	 */
	
	public boolean canAdminister(String userid) 
	{
		boolean retval = false;
		
		try
		{
				PersistentUser user = new PersistentUser(userid);
				user.fetch();
				int userpriv = user.getPriv();
				if (0 !=(userpriv & DefectPrivileges.CANADMINISTER))
					retval = true;
		}
		
		catch (SQLException sqle)
		{
			;// we'll just return false
		}
		
		return retval;
		
	}
	
	/**
	 Returns the next sequential defect number.
	 @return the next available, sequential defect id
	*/
	public synchronized int getNextDefectID()
	{
		m_nextDefectID++;
		return m_nextDefectID;
	}
	
	public void setOrb(org.omg.CORBA.ORB orb)
	{
		m_Orb = orb;
	};
	
	public void implConnect(org.omg.CORBA.Object obj)
	{
		m_Orb.connect(obj);
	}
	
	public void implDisconnect(org.omg.CORBA.Object obj)
	{
		m_Orb.disconnect(obj);
	}
	
	/**
	
	@param username id of user making the changes
	@exception AppSecurityException thrown if appropriate privileges don't exist for username
	*/
	public void securityCheck(PersistentUser user, String username) throws AppSecurityException
	{
			if (!canAdminister(username))
				throw new AppSecurityException();
		
	 }
	
	/** 
	this method checks to ensure that changes made to the 
	defect are permitted by the defect change policies
	@param defect a referece to the user to be changed
	@param username id of user making the changes
	@exception AppSecurityException thrown if appropriate privileges don't exist for username
	*/
	public void securityCheck(PersistentDefect defect, String username) throws AppSecurityException
	{
		
		boolean stateChangedToVerified, stateChangedToResolved;
		boolean priorityChanged, assignmentChanged;
			
		// is this a new defect or already exists?
		PersistentDefect stored = new PersistentDefect();
		stored.setDefectID(defect.getDefectID());
		
		stateChangedToVerified = false;
		stateChangedToResolved = false;
		priorityChanged = false;
		assignmentChanged = false;
		
		if (stored.rowExists())
		{
			try
			{
				stored.fetch();
			}
			catch (SQLException sqle)
			{
				/* not the most elegant way to handle,
				but if you wind up here it's fairly
				nasty, so just bail.
				*/
				throw new AppSecurityException();
				
				}			
			
			// check the fields that are secured
			
			if (!(stored.getAssignedTo().trim().equals(defect.getAssignedTo().trim())))
				assignmentChanged = true;
			if (stored.getPriority() != defect.getPriority())
				priorityChanged = true;
			if (stored.getState() != defect.getState())
				if (defect.getState() == DefectState.FIX_VERIFIED)
					stateChangedToVerified = true;
				else
					if (defect.getState() == DefectState.FIXED)
						stateChangedToResolved = true;
		}
		else
		{
			if (defect.getAssignedTo().trim().length() > 0)
				assignmentChanged = true;
			if (defect.getPriority() != DefectPriority.NULL)
				priorityChanged = true;
			if (defect.getState() == DefectState.FIX_VERIFIED)
				stateChangedToVerified = true;
			else if (defect.getState() == DefectState.FIXED)
				stateChangedToResolved = true;
		}
		
		if ( assignmentChanged )
			if ( !canAssign(username))
				throw new AppSecurityException();
				
		if ( priorityChanged )
			if ( !canPrioritize(username))
				throw new AppSecurityException();
				
		if ( stateChangedToVerified )
		{
			if (!canVerify(username))
				throw new AppSecurityException();
		}
		else if ( stateChangedToResolved )
			if (!canResolve(username))
				throw new AppSecurityException();
					
		
	}
	
	
	/**
	Method for supporting the TestHarness
	*/
	public boolean TestUnit()
	{
		boolean retval = false;
		String userid = "rkp";
		
		
		DefectPolicy policy = new DefectPolicy();
		
		int first = policy.getNextDefectID();
		int next = policy.getNextDefectID();
		
		if (first < next)
			if (policy.canAssign(userid))
				if (policy.canResolve(userid))
					if (policy.canVerify(userid))
						if (policy.canPrioritize(userid))
							retval = true;
		
		return retval;
		
	}
	
	
	
}