//Source file: c:/work/wrox/case_study/code/BugTrackerApp/dataabstraction/PersistentUser.java

package dataabstraction;

import businesslogic.*;
import businessmodel.*;
import apputil.*;


import java.sql.*;
import java.util.*;

/**
The PersistentUser class provides the object-relational / relational-object mapping
for User objects in the BugTracker application. It handles all persistence issues for 
User objects. 
The User objects that are remoted to the BugTracker application and servlet are
of the PersistentUser class. 
*/
public class PersistentUser extends businessmodel.User implements Persistent, apputil.Testable
{
	
	static Connection databaseConnection = null;
  static String driverName;
  static String sourceURL;
	  
  public static final String QUERYBYKEY = "select userid, lastname, firstname, phonenumber, faxnumber, emailaddress, priv_levels from users where userid = ?";
	public static final String DROPBYKEY = "delete from users where userid = ?";
	public static final String INSERTUSER = "insert into users (userid, lastname, firstname, phonenumber, faxnumber, emailaddress, priv_levels) values (?,?,?,?,?,?,?)";
	public static final String UPDATEUSER = "update users set lastname=?, firstname=?, phonenumber=?, faxnumber=?, emailaddress=?, priv_levels=? where userid = ?";
	public static final String GETUSERIDS = "select userid from users order by userid";
	
	/**
	This method ensures that the JDBC driver is loaded and
	we have a connection to the database. The first time it's
	called, it gets the driver name and datasource URL from the
	application properties and loads the database connection. 
	Subsequent calls to this method are essentially no-ops
	since the connection has already been established.
	*/
	protected void getDatabaseConnection() throws SQLException
	{
		
		if (databaseConnection == null)
		{
	
			AppConfig config = new AppConfig();
		
			try
			{
				driverName = config.getDatasourceDriver();
  			sourceURL = config.getDatasourceURL();
	 			Class.forName (driverName);
  			databaseConnection = DriverManager.getConnection(sourceURL);
			}
			catch(ClassNotFoundException cnfe)
			{
				throw new SQLException("Invalid driver specified in application properties");
			}
		}
 
	}
	
	
	/**
	Default constructor. Ensures that there is a connection to the data source.
	*/
	public PersistentUser() 
	{
		
		super();
	  try
		{
			getDatabaseConnection();
		}
		catch(SQLException sqle)
		{
			throw new RuntimeException("failed in PersistentUser::PersistentUser()\n" 
				+ sqle.toString());
		
		 }
	}

	/**
	Constructor. Assigns user id, ensures that there is a connection to the data source.
	*/
	public PersistentUser(String uid) 
	{
		
		super();
	  try
		{
			getDatabaseConnection();
		}
		catch(SQLException sqle)
		{
			throw new RuntimeException("failed in PersistentUser::PersistentUser()\n" 
				+ sqle.toString());
		
		 }
		 
		 setUserID(uid);
		 
	}


	
	/**
	Fetch the user object from the persistent store based on the user id
	@exception java.sql.SQLException 
	 */
	public void fetch() throws java.sql.SQLException 
	{
		String theUserID = this.getUserID();
		
		if (theUserID == null)
		{
			throw new SQLException("Cannot fetch user without id set");
		}
		
		PreparedStatement pQuery = 
				databaseConnection.prepareStatement(QUERYBYKEY);
		pQuery.setString(1, getUserID());
		
		ResultSet rs = pQuery.executeQuery();
		
		//any returns?
		if (rs.next())
		{
			
			String stringval;
			int intval;
			
			stringval = rs.getString("userid");
			setUserID(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("lastname");
			setLastName(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("firstname");
			setFirstName(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("phonenumber");
			setPhone(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("faxnumber");
			setFax(rs.wasNull()?"":stringval);
			
			stringval = rs.getString("emailaddress");
			setEmail(rs.wasNull()?"":stringval);
			
			intval = rs.getInt("priv_levels");
			setPriv(rs.wasNull()?0:intval);
			rs.close();
		}
		
		else
			throw new SQLException("No matching rows found!");
		
	}
	
	/**
	Delete the data from the persistent store for this user.
		exception java.sql.SQLException 
	*/
	public void drop() throws java.sql.SQLException 
	{
		String theUserID = this.getUserID();
		
		if (theUserID == null)
		{
			throw new SQLException("Cannot drop user without id set");
		}
		
		PreparedStatement pQuery = 
				databaseConnection.prepareStatement(DROPBYKEY);
		pQuery.setString(1, getUserID());
		pQuery.executeUpdate();
			
	}
	
	/**
	Insert the user information into the database as a new user
	@exception java.sql.SQLException
	 */
	public void insert() throws java.sql.SQLException 
	{
		String theUserID = this.getUserID();
		
		if (theUserID == null)
		{
			throw new SQLException("Cannot add user without id set");
		}
		
		PreparedStatement pQuery = 
				databaseConnection.prepareStatement(INSERTUSER);
		
		pQuery.setString(1, getUserID());
		pQuery.setString(2, getLastName());
		pQuery.setString(3, getFirstName());
		pQuery.setString(4, getPhone());
		pQuery.setString(5, getFax());
		pQuery.setString(6, getEmail());
		pQuery.setInt(7, getPriv());
	
		pQuery.executeUpdate();
		
	}
	
	/**
		Update the user information in the database
		@exception java.sql.SQLException
		 */
	public void update() throws java.sql.SQLException 
	{
		String theUserID = this.getUserID();
		if (theUserID == null)
		{
			throw new SQLException("Cannot update user without id set");
		}
		
		PreparedStatement pQuery = 
				databaseConnection.prepareStatement(UPDATEUSER);
		
		
		pQuery.setString(1, getFirstName());
		pQuery.setString(2, getLastName());
		pQuery.setString(3, getPhone());
		pQuery.setString(4, getFax());
		pQuery.setString(5, getEmail());
		pQuery.setInt(6, getPriv());
		pQuery.setString(7, getUserID());
		
		pQuery.executeUpdate();
		
		
	}
	
	/**
	indicates whether the user is stored in the database or not
	@return boolean value indicating whether the user exists in the database or not
	*/
	public boolean rowExists()
	{
		
		boolean retval;
		
		PersistentUser pd = new PersistentUser();
		pd.setUserID(getUserID());
		try
		{
			pd.fetch();
			retval = true;
		}
		catch (SQLException sqle)
		{
			retval = false;
		}
		return retval;
		
	}

	
	/**
	Attempt to store updates made to a user record
	@param userid the identity of the user requesting the change to be made
	@exception UpdateException thrown if the data can't be written
	@exception AppSecurityException thrown if the user doesn't have administration privileges		
	*/	
	public void commit(String userid) throws UpdateException, AppSecurityException
	{
		
		super.commit(userid);
		
		try
		{
			if (rowExists())
				update();
			else
				insert();
		}
		catch(SQLException sqle)
		{
			throw new UpdateException();
		}
	};		

	/**
	get all user ids
	@return returns a String array containing all user ids
	*/
	public String[] getUserIDList()
	{
		
		String[] retval;
		
		try
		{				
			Statement pQuery = 
					databaseConnection.createStatement();
			
			ResultSet rs = pQuery.executeQuery(GETUSERIDS);
			
			Vector userIdVector = new Vector();
			
			//any returns?
			while (rs.next())
			{
				userIdVector.addElement(rs.getString("userid").trim());
			}
			rs.close();
			
			int numUsers = userIdVector.size();
			retval = new String[numUsers];
			retval = (String[])userIdVector.toArray(retval);
		}
		catch(SQLException sqle)
		{
			// if this blew up, just return an empty array
			retval = new String[0];
						
		}
		
		return retval;
		
	}


	
	/**
	 Debug method; provides a text representation of the internal User state
	 	 */
	public String toString() 
	{
		String retval = new String("PersistentUser{" + 
			", Driver = " + driverName +
			", URL = " + sourceURL + 
			super.toString()) + "}";
			
			return retval;
		}
	
	/**
	TestUnit implemented to support test harness
	*/
	
	public boolean TestUnit()
	{
		
		boolean retval;
		String newFax = new String("999.999.9999");
		PersistentUser test = new PersistentUser();
				
		retval = super.TestUnit();
		if (retval)
		{
			retval = false;
			try
			{
				this.insert();
				test.setUserID(this.getUserID());
				test.fetch();
				
				if(test.getUserID().equals(this.getUserID()))
					if(test.getLastName().equals(this.getLastName()))
						if(test.getFirstName().equals(this.getFirstName()))
							if(test.getPhone().equals(this.getPhone()))
								if(test.getFax().equals(this.getFax()))
									if(test.getEmail().equals(this.getEmail()))
										if(test.getPriv() == this.getPriv())
											retval = true;
				
				if (retval)
				{
					this.setFax(newFax);
					this.update();
					
					// test the getUserIDList method
					String[] users = this.getUserIDList();
					
					int UserCount = users.length;
					System.out.println("user count = " + UserCount);
					
					// iterate through the user ids and print them out
					if (UserCount > 0)
					{
						int nPos = 0;
					
						while (nPos < UserCount)
							System.out.print(users[nPos++] + " ");
					
						System.out.println("");	
					}
					else
						return false; // error, should have a valid list of users
					
					this.fetch();
					if (this.getFax().equals(newFax))
					{
						this.drop();
						try
						{
							this.fetch();
							retval = false;
						}
						catch (SQLException sqle)
						{
							// should have gotten here, since row is gone
							retval = true;
						}
					}
				}
			}		
			catch (SQLException sqle)
			{
				System.out.println(sqle.toString());
				retval = false;
			}
		}
		
		return retval;
		
	 }
}