import businesslogic.*;
import businessmodel.*;

import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.*;
import org.omg.CORBA.*;
import org.omg.CosNaming.*;


/**
Servlet that provides a Web-based interface to the BugTracker application
*/
public class BugTrackerServlet extends HttpServlet
{
	
	/**
	A single user called "www" has been created as an agent for web-submitted
	defects. This user was created in the build_tables script for this
	application.
	*/
	static final String WWWUSER = "www";
	static final String NULLTEXT = "<not specified>";
	
	DefectServerIntf m_DefectServer;
	DefectFactoryIntf m_DefectFactory;
	DefectPolicyIntf m_DefectPolicy;
	ORB m_Orb;


	/**
	init method sets up the applet by initializing the ORB and getting
	references to the BugTracker server resources.
	*/ 
	public void init(ServletConfig config) throws ServletException
	{
			super.init(config);
			
			System.out.println("trying to initialize orb...");
			try
			{
				  String propsfile = config.getInitParameter("propsfile");
				  
				  apputil.AppConfig appcfg;
					if (propsfile == null)
				  	appcfg = new apputil.AppConfig("c:\\work\\wrox\\case_study\\code\\BugTrackerApp\\BUGTRACKER.PROPS");
					else
						appcfg = new apputil.AppConfig(propsfile);
									
					Properties props = new Properties();
					
					String host = config.getInitParameter("ORBInitialHost");
					if (host != null)
						props.setProperty("ORBInitialHost", host);
						
					String port = config.getInitParameter("ORBInitialPort");
					if (port != null)
						props.setProperty("ORBInitialPort", port);	
					
			    // create and initialize the ORB
			    m_Orb = ORB.init((String[])null, props);
		 
		      // get the root naming context
		      org.omg.CORBA.Object objRef = m_Orb.resolve_initial_references("NameService");
		      NamingContext ncRef = NamingContextHelper.narrow(objRef);
		
		      // resolve the Object Reference in Naming
		      NameComponent nc = new NameComponent(appcfg.getBugServerServiceName(), "");
		      NameComponent path[] = {nc};
		      
		      //Get a reference to the bug server
		      m_DefectServer = DefectServerIntfHelper.narrow(ncRef.resolve(path));
					m_DefectFactory = m_DefectServer.getDefectFactory();
					m_DefectPolicy = m_DefectServer.getPolicyObject();	
		      
			}
			catch(Exception e)
			{
				System.out.println(e.toString());	
					throw new ServletException(e.toString());
				
			}
		
	}
	
	/**
	This method receives all POST requests. This is a dispatcher method
	that routes the request to the appropriate method, based on the type
	of request as identified in the parameters sent.
	@param req the request object
	@param resp the response object
	*/
	public void doPost(HttpServletRequest req, HttpServletResponse resp)
	{
		
		String reqType = req.getParameter("reqtype").toUpperCase();
		
		if (reqType.equals("SUBMIT"))
			handleSubmit(req, resp);
		else 
		if (reqType.equals("QUERY"))
			handleQuery(req, resp);
		else
			handleBadRequest(req, resp);	
		
	}

	
	/**
	Handle the post method from the finddefect.html form. Called as a 
	dispatch method of doPost. 
	The form provides the following fields
	
	defectid	:	the defect id to look up
	format		:	either "html" or "xml"
	
	*/
	
	public void handleQuery(HttpServletRequest req, HttpServletResponse resp)
	{
		
		try
		{
			
			String defectId = null;
			String xmlString;
			String[] defectIdList;
			String retBody;
			String format = new String("html");
			
			boolean gotDefectId = false;
			
			// get the writer object 
			PrintWriter out = resp.getWriter();
						
			Enumeration enum;
			String parmName;
			
			enum = req.getParameterNames();
			while (enum.hasMoreElements())
			{
				parmName = (String)enum.nextElement();
				
				// is it the defect ids?
				if (parmName.trim().equals("defectid"))
				{
					// got a defect request
					defectId = req.getParameter(parmName); 
					gotDefectId = true;
					
				 }
				else if (parmName.trim().equals("format"))
				{
					format = req.getParameter(parmName);
						
					// make sure it's correct case
					format = format.toLowerCase();	
				} 
			}
			
			resp.setContentType("text/" + format);
			
			if (gotDefectId)
			{
				if (format.equals("xml"))
					retBody = OutputAsXML(defectId);
				else
					retBody = OutputAsHTML(defectId);
			}
			else
				retBody = "no defect specified";
				
			out.println(retBody);
			out.close();
		}
		
		catch(IOException ioe)
		{
			; // oh well...
			}
		
	}
			
			
	/**
	Handle the post method from the submitdefectreport.html form. That form
	provides the following fields that must be read here, and formulated into 
	a defect:
	<pre>
	defectsummary	:	use value in DefectIntf.setDefectSummary
	defectdetails	:	the user's description of the problem
	name	:	these items will be formatted and concatenated into the
	fax			defect details
	phone
	company
	email
	productid
	notifyonresolve	:	if this comes back true, then the email address 
										value will be sent to the DefectIntf.setResolvedMailList method
	product	:	use value in DefectIntf.setProduct
	version	: use value in DefectIntf.setVersionReported
	</pre>
	*/		
	public void handleSubmit(HttpServletRequest req, HttpServletResponse resp)
	{
		resp.setContentType("text/html");
		String retval = new String("");
		String errorText;
		int defectId;
			
		try
		{
			
			PrintWriter out = resp.getWriter();
  		DefectIntf theDefect;
			
			
			try
			{
				String defectSummary	= req.getParameter("defectsummary");
				if (defectSummary == null)
					throw new BadInformationException("Defect summary is required");
					
				String defectDetails	= req.getParameter("defectdetails");
				if (defectDetails == null)
					throw new BadInformationException("Defect detail is required");

				String name	= req.getParameter("name");
				if (name == null)
					name = NULLTEXT;
					
				String phone = req.getParameter("phone");
				if (phone == null)
					phone = NULLTEXT;
					
				String fax = req.getParameter("fax");
				if (fax == null)
					fax = NULLTEXT;
				
				String company = req.getParameter("company");
				if (company == null)
					company = NULLTEXT;
	
				boolean notifyonresolve
					= (req.getParameter("notifyonresolve").trim().equals("1"));
				
				String email = req.getParameter("email");
				if (email == null)
					throw new BadInformationException("Email address is required");
	
				String product = req.getParameter("product");
				if (product == null)
					throw new BadInformationException("Product name is required");

				String version = req.getParameter("version");
				if (version == null)
					version = NULLTEXT;
					
				String productid = req.getParameter("productid");
				if (productid == null)
					productid = NULLTEXT;
										
				/*
				If you have more validation rules that you'd like to add, such
				as ensuring a valid serial number, etc, do so here. Throw
				a BadInformationException to stop the defect submitting process.
				*/	
				
				theDefect = m_DefectFactory.NewDefect(WWWUSER);
				defectId = theDefect.getDefectID();
				
				String appendText = new String("\nSubmitted by: " + name +
					"\nCompany: " + company +
					"\nPhone: " + phone + 
				 	"\nFax: " + fax +
				 	"\nEmail: " + email +
				 	"\nProduct ID: " + productid);
				
				theDefect.setState(DefectState.NEW); 
				theDefect.setReportedBy(WWWUSER);		
				theDefect.setDefectSummary(defectSummary);
				theDefect.setDefectDetails(defectDetails + appendText);
				theDefect.setProduct(product);
				theDefect.setVersionReported(version);
				if (notifyonresolve)
					theDefect.setResolvedMailList(email);
				
				theDefect.commit(WWWUSER);	
				
				retval = retval.concat("<head><title>Defect submitted</title></head>");
				retval = retval.concat("<body><h1>Defect Submitted</h1>");
				retval = retval.concat("<hr>Your report has been submitted as defect <b>" + defectId);
				retval = retval.concat("</b><p>Please refer to this number when inquiring on the status of ths defect");

			}
			catch(BadInformationException bie)
			{
				
				errorText = bie.toString();
				retval = retval.concat("<head><title>Error submitting defect</title></head>");
				retval = retval.concat("<body><h1>Defect Not Submitted</h1>");
				retval = retval.concat("<hr>The following error occurred:<p>" + errorText);

			}
			catch(UpdateException ue)
			{
				errorText = ue.toString();
				retval = retval.concat("<head><title>Error submitting defect</title></head>");
				retval = retval.concat("<body><h1>Defect Not Submitted</h1>");
				retval = retval.concat("<hr>The following error occurred:<p>" + errorText);
				
			}
			catch(AppSecurityException ase)
			{
				errorText = ase.toString();
				retval = retval.concat("<head><title>Error submitting defect</title></head>");
				retval = retval.concat("<body><h1>Defect Not Submitted</h1>");
				retval = retval.concat("<hr>The following error occurred:<p>" + errorText);
	
			}
			
			retval = retval.concat("</body></html>");
			out.println(retval);
			out.close();

		}
		catch (IOException ioe)
		{
			;
		}
	}
	

	/**
	Handler routine for bad/invalid Post requests
	*/
	public void handleBadRequest(HttpServletRequest req, HttpServletResponse resp)
	{
			resp.setContentType("text/html");
	
			try
			{
				
				PrintWriter out = resp.getWriter();
				out.println("Bad request to servlet... see system administrator");
				out.close();
			}
			catch(IOException IOE)
			{
				
			}
	
	}

	public String getServletInfo()
	{
		return "BugTrackerApp servlet";
	}
	
	/**
	Return the list of defects into HTML. 
	Because each defect's transformation to XML is a complete, well
	formed document, this limits the number of returned defects to one.
	Multiple calls are required to get more than one defect in XML.
	*/			
	String OutputAsXML(String defectId)
	{
		DefectIntf theDefect;
		String retval;
		
		try
		{
			int id;
			id = Integer.parseInt(defectId);
			theDefect = m_DefectFactory.getDefect(id);
			retval = theDefect.exportToXML();	
		}
		
		catch(NumberFormatException numfe)
		{
			retval = new String(defectId + 
				" is not a properly formatted defect id.");
		}
		
		catch(NotFoundException nfe)
		{
			retval = new String(defectId + 
				" is not a valid defect.");
				
		}
		
		return retval;
		
	}

	/**
	Format the list of defects into HTML. 
	This is pretty much grunt work here, but there are better ways
	to do this. 
	
	This would be a great place to get the defect as XML using the 
	DefectIntf.exportToXML method, and use XSL to format it to HTML. 
	There are tools that are freely available to do this parsing &
	transformation; see IBM AlphaWorks (www.alphaworks.ibm.com) 
	for some really effective tools.
	*/	
	String OutputAsHTML(String defectId)
	{
		DefectIntf theDefect;
		int n;
		
		String retval = new String("");
		
		retval = retval.concat("<html><head><title>Defect Report</title></head>");
		retval = retval.concat("<body><h1>BugTracker Defect Report</h1>");
		retval = retval.concat("<hr><p>");
		try
		{
			int id;			id = Integer.parseInt(defectId);
			theDefect = m_DefectFactory.getDefect(id);
			
			retval = retval.concat("<b>Defect ID:</b>" + theDefect.getDefectID() + "<br>");
				
			retval = retval.concat("<b>Defect Status:</b> " + 
				DefectState.stateNames[theDefect.getState()] + "<br>");
				
			retval = retval.concat("<b>Date Reported:</b> " + 
				apputil.util.LongToDateString(theDefect.getDateReported()) + "<br>");
				
			retval = retval.concat("<b>Defect Summary:</b> " + theDefect.getDefectSummary() + "<br>");
			
			retval = retval.concat("<b>Defect Details:</b><pre> " + theDefect.getDefectDetails() + "</pre>");
				
			retval = retval.concat("<b>Date of last action:</b> " + 
				apputil.util.LongToDateString(theDefect.getDateLastChanged()) + "<br>");
					
		}
			
		catch(NumberFormatException numfe)
		{
			retval = retval.concat("<b>" + defectId + 
				" is not a properly formatted defect id.<b>");
		}
		
		catch(NotFoundException nfe)
		{
			retval = retval.concat(defectId + 
				" is not a valid defect.");
				
		}

		retval = retval.concat("</body></html>");
		return retval;
	}	
		
}


/**
Exception for handling bad submitted defect information
*/

class BadInformationException extends Exception
{
		public BadInformationException()
		{
			super();	
		}
		
		public BadInformationException(String s)
		{
			super(s);	
		}
}
	